/* fton.c
 * Copyright (C) 1998 N.Fukase
 * Modified by William Stein (was@math.berkeley.edu), November 1999.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA
 */

/*
** original ftr 0.1a from http://www.flatout.org/~fukase/uxrup.html
**     mailto:fukase@flatout.org (N.Fukase)
** fton 0.1 (for OnHand) from http://modular.fas.harvard.edu/onhand/ 
**     mailto:was@math.harvard.edu (William Stein) 
**
** 10-March-2002 clach04 Chris.Clark@ca.com
**  Increased version to 0.2
**  Added ifdef's around readline dependent code and added quick
**  poor-mans readline function in it's place. They may be memory leaks.
**  Added baud rate command line option for dual Ruputer/OnHand usage.
**    NB OnHand doesn't appear to want to do anything other than 38400!
**    Service pack options for FileTrans have no effect.
**  Added simple usage output.
**  Added CTRL-C message to the connecting code.
**  Added aliases for "bye" and "exit" - more ftp like.
**  Added aliases for "1ls", !dir" - again more ftp like.
**  Added "dir" which is like ls -l, but not 100% the same:
**      does filesize, 
**      displays filename in lowercase,
**      displays date/time (year/month/day 24 hour time)
**      attempts to display unix like file perms, added "h" for hidden/system
**  Added environment variables (command line params overwrde) 
**          FTON_BAUDRATE   -- baud rate
**          FTON_SERIALPORT -- serial device to use
**
**  Know bugs - no params given after -l/-b - core dump do to access to argv
**   that does not exist, the command line params could be better handled
**   by getopt.
**  Builds under win32 with cygwin with or without readline.
**  Default is NOT to use readline, #define USE_READLINE if required
**  Default serial device is /dev/ttyS0 (i.e. com1)
**  Default baud rate is OnHand rate of 38400
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <time.h>
#include <utime.h>

#ifdef USE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "rupcommand.h"
#include "rupserial.h"

static char	rup_line[FILENAME_MAX] = {"/dev/ttyS0"};
static char	rup_cwd[64] = {"B:\\"};
static char	rup_cwdA[64] = {"A:\\"};
static char	rup_cwdB[64] = {"B:\\"};
static char	prompt[80]= {"fton> "};
static char	ftr_version[]= {"0.2"};

/* readline function defs */
#ifndef USE_READLINE
typedef int Function ();
#endif

int	rlc_ls(char **coms);
int	rlc_lsl(char **coms);
int	rlc_ver(char **coms);
int	rlc_ren(char **coms);
int	rlc_cp(char **coms);
int	rlc_mkdir(char **coms);
int	rlc_rmdir(char **coms);
int	rlc_del(char **coms);
int	rlc_lpwd(char **coms);
int	rlc_lcd(char **coms);
int	rlc_pwd(char **coms);
int	rlc_put(char **coms);
int	rlc_get(char **coms);
int	rlc_lls(char **coms);
int	rlc_cd(char **coms);
int	rlc_help(char **coms);

typedef enum {
     ca_none = 0,
     ca_remote,
     ca_remote_d,
     ca_local,
     ca_local_d,
     ca_line,
     ca_command,
     ca_valiable
} COMATTR;

typedef struct {
     char	*name;
     COMATTR	attr;
     Function	*func;
     char	*doc;
} COMMAND;

COMMAND commands[] = {
     { "bye", ca_none, NULL, "quit this programm"},
     { "cd", ca_remote_d, rlc_cd, "change onHand\'s current working directory"},
     { "close", ca_none, NULL, "close the connection to onHand"},
     { "cp", ca_remote_d, rlc_cp, "copy file in  onHand\'s"},
     { "delete", ca_remote, rlc_del, "delete file in onHand"},
     { "dir", ca_remote, rlc_lsl, "list long contents of onHand\'s directory" },
     { "exit", ca_none, NULL, "quit this programm"},
     { "get", ca_remote, rlc_get, "get a file from onHand" },
     { "help", ca_command, rlc_help, "display this text" },
     { "?", ca_command, rlc_help, "display this text"},
     { "!ls", ca_local, rlc_lls, "list contents of local directory"},
     { "!dir", ca_local, rlc_lls, "list contents of local directory"},
     { "lcd", ca_local_d, rlc_lcd, "change local current working directory"},
     { "lls", ca_local, rlc_lls, "list contents of local directory"},
     { "lpwd", ca_none, rlc_lpwd, "print local current working directory" },
     { "ls", ca_remote, rlc_ls, "list contents of onHand\'s directory" },
     { "mkdir", ca_remote, rlc_mkdir, "make directory in onHand" },
     { "mv", ca_remote, rlc_ren, "rename file in onHand" },
     { "open", ca_none, NULL, "open a connection to onHand"},
     { "put", ca_local, rlc_put, "put a file into onHand" },
     { "pwd", ca_none, rlc_pwd, "print the current working directory of onHand"},
     { "quit", ca_none, NULL, "quit this programm"},
     { "exit", ca_none, NULL, "quit this programm"},
     { "rename", ca_remote, rlc_ren, "rename file"},
     { "rm", ca_remote, rlc_del, "remove file in onHand" },
     { "rmdir", ca_remote, rlc_rmdir, "remove a directory in onHand" },
     { "ver", ca_none, rlc_ver, "print WPs-DOS\'s version"},
     { (char *)NULL, ca_none, (Function *)NULL, (char *)NULL }
};

char *
ruputerfile_generator(char *text, int state)
{
     static int		list_index, len;
     static RDENT	rde = {0, NULL, ""};

     if (!state) {
	  list_index = -1;
	  len = strlen(text);
	  free(rde.rfi);
	  strcpy(rde.path, rup_cwd);
	  strcat(rde.path, "*.*");
	  command_dir(&rde);
     }
     while ((list_index + 1) < rde.nfiles) {
	  list_index++;
	  if (strncmp(rde.rfi[list_index].name, text, len) == 0) {
	       return strdup(rde.rfi[list_index].name);
	  }
     }
     return ((char *)NULL);
}

char *
command_generator (char *text, int state)
{
     static int	list_index, len;
     char	*name;
     
     if (!state) {
	  list_index = 0;
	  len = strlen(text);
     }
     while ((name = commands[list_index].name) != NULL) {
	  list_index++;
	  if (strncmp (name, text, len) == 0)
	       return strdup(name);
     }
     return ((char *)NULL);
}

#ifdef USE_READLINE
char **
rft_comp(char *text, int start, int end)
{
     char	**matches;
     char	buf[64];
     int	i;
     
     matches = (char **)NULL;
     if (start == 0)
	  matches = completion_matches(text, command_generator);
     else {
	  strcpy(buf, rl_line_buffer);
	  if(strtok(buf, " \t") != NULL) {
	       for(i = 0; commands[i].name != NULL; i++)  {
		    if (strcmp(commands[i].name, buf) == 0) {
			 if((commands[i].attr == ca_remote) || (commands[i].attr == ca_remote_d)) {
			      matches = completion_matches(text, ruputerfile_generator);
			      break;
			 } else if(commands[i].attr == ca_command) {
			      matches = completion_matches(text, command_generator);
			      break;
			 }
		    }
	       }
	  }
     }
     return (matches);
}
#endif

void
init_readline(void)
{
#ifdef USE_READLINE
     rl_readline_name = "rft";
     rl_attempted_completion_function = (CPPFunction *)rft_comp; 
#endif
}

char *
myreadline(char * txt)
{
  char * ret_str;
   ret_str = malloc (sizeof(char) * 1024);
  printf("%s", txt);
  gets(ret_str);
 return ret_str;
}

void
intteruct(void)
{
     char	*coms[256], *name, *line;
     int	i;

     init_readline();
     for(;;) {
#ifdef USE_READLINE
	  line = readline(prompt);
#else
	  line = myreadline(prompt);
#endif
	  if (line && *line) {
#ifdef USE_READLINE
	       add_history(line);
#endif
	       if((coms[0] = strtok(line, " \t")) == NULL) {
		 free(line);
		 continue;
	       }
	       for (i = 1; (coms[i] = strtok(NULL, " \t")) != NULL; ++i)
		    ;

	       for (i = 0; (name = commands[i].name) != NULL; i++) {
		    if (*coms[0] == 'q' || *coms[0] == 'e' || *coms[0] == 'b') {
			 free(line);
			 return;
		    }
		    if (strcmp(name, coms[0]) == 0) {
			 if(commands[i].func != NULL)
			      (*(commands[i].func))(coms);
			 else
			       printf("Sorry! this command is not implemented yet\n");
			 break;
		    }
	       }
	       if (name == NULL)
		    printf("no such command `%s\'\n", coms[0]);
	  }
	  free(line);
     }
}

/* Onhand coms routines */
void
send_file(char *fn, char *rp)
{
     char		*ptr;
     int		fd;
     LENDAT		dat; /* žǡǼ	*/
     RFINFO		rfi; /* žǡ		*/
     struct stat	st;
     struct tm		*ftm;

     if ((fd = open(fn, O_RDONLY)) >= 0) {
	  fstat(fd, &st);
	  if ((dat.data = (char *)malloc(st.st_size)) == NULL) {
	       fprintf(stderr, "Cannot allocate memory\n");
	       close(fd);
	  } else {
	       read(fd, dat.data, st.st_size);
	       dat.length = st.st_size;
	       ftm = localtime(&(st.st_mtime));
	       rfi.cdate = ((((ftm->tm_year - 80) & 0x7f) << 9) | \
			    (((ftm->tm_mon & 0xf) + 1) << 5) | \
			    (ftm->tm_mday & 0x1f));
	       rfi.ctime = (((ftm->tm_hour & 0x1f) << 11) | \
			    ((ftm->tm_min & 0x3f) << 5) | \
			    ((ftm->tm_sec >> 1) & 0x1f));
	       rfi.attr = 0;
	       rfi.size = st.st_size;
	       close(fd);
	       if((ptr = strrchr(fn, '/')) == NULL) {
		    strcpy(rfi.name, fn);
	       }else {
		    strcpy(rfi.name, ptr + 1);
	       }
	       command_upload(&dat, &rfi, rp);
	       free(dat.data);
	  }
     } else
	  fprintf(stderr, "Cannot open file %s\n", fn);
}

void
recv_file(char *fn, char *rp)
{
     int		fd;
     LENDAT		dat; /* žǡǼ	*/
     RFINFO		rfi; /* žǡ		*/
     struct tm		ftm;
     struct utimbuf	ut;

     if ((fd = open(fn, O_RDWR | O_CREAT | O_EXCL, 0777)) >= 0) {
	  strncpy(rfi.name, fn, 12);
	  rfi.name[13] = '\0';
	  if (command_download(&dat, &rfi, rp) != 0) {
	       close(fd);
	       return;
	  }
	  write(fd, dat.data, dat.length);
	  close(fd);
	  ftm.tm_year = ((rfi.cdate >> 9) & 0x7f) + 80;
	  ftm.tm_mon = ((rfi.cdate >> 5) & 0xf) - 1;
	  ftm.tm_mday = (rfi.cdate & 0x1f);
	  ftm.tm_hour = ((rfi.ctime >> 11) & 0x1f);
	  ftm.tm_min = ((rfi.ctime >> 5) & 0x3f);
	  ftm.tm_sec = ((rfi.ctime & 0x1f) << 1);
	  ftm.tm_isdst = 0;
	  ut.actime = mktime(&ftm);
	  ut.modtime = ut.actime;
	  utime(rfi.name, &ut);
	  free(dat.data);
     } else
	  fprintf(stderr, "Already have file %s\n", fn);
}
/*_____ ls _________________________________________________________________*/
void
get_file_list(char *rf, char *rp)
{
     RDENT	rde;
     int	i;

     strcpy(rde.path, rp);
     if (rf == NULL)
	  strcat(rde.path, "*.*");  
     else
	  strcat(rde.path, rf);  
     command_dir(&rde);

     for(i = 0; i < rde.nfiles; ++i) {
	  printf("%s\n", rde.rfi[i].name);
     }
     free(rde.rfi);
}

char *
decode_filedate(unsigned short ohdate)
{
  static char outstr[20];
  int day, month, year;

  day = ohdate & 0x1f;
  month = ((ohdate >> 5) & 0xf); /* real month num, e.g. 1 = jan */
  year = ((ohdate >> 9) & 0x7f) + 1980;

  sprintf(outstr, "%d/%2.2d/%2.2d", year, month, day);
  return outstr;
} 

char *
decode_filetime(unsigned short ohdate)
{
  static char outstr[20];
  int sec, min, hour;

  sec = (ohdate & 0x1f) * 2;
  min = ((ohdate >> 5) & 0x3f);
  hour = ((ohdate >> 11) & 0x1f);

  sprintf(outstr, "%2.2d:%2.2d:%2.2d", hour, min, sec);
  return outstr;
}



char *
decode_fileattr(unsigned char inatt)
{
  static char outstr[12];

  strcpy (outstr, "-rwxrwxrwx ");
  /* directory */
  if (inatt & 0x10) outstr[0] = 'd'; /* directory */

  /* file */
  if (inatt & 0x20) outstr[0] = '-'; /* file */

  /* hidden */
  if (inatt & 0x02) outstr[10] = 'h'; /* hidden */
  return outstr;
}

char *
filename_case(char * inname)
{
static char outname[FILENAME_MAX];
int count = 0;

while (inname[count] != '\0' )
{
  outname[count]  = tolower(inname[count]);
  count++;
}

outname[count]  ='\0'; 
return outname;
}

void
get_file_listlong(char *rf, char *rp)
{
     RDENT      rde;
     int        i;

     strcpy(rde.path, rp);
     if (rf == NULL)
          strcat(rde.path, "*.*");
     else
          strcat(rde.path, rf);
     command_dir(&rde);

     for(i = 0; i < rde.nfiles; ++i) {
          printf("%s - %s %s %10.0d %s\n", 
decode_fileattr(rde.rfi[i].attr), 
decode_filedate(rde.rfi[i].cdate), decode_filetime(rde.rfi[i].ctime),
rde.rfi[i].size, filename_case(rde.rfi[i].name));

/*
 printf("%#2.2x  %d %s\n", rde.rfi[i].attr, rde.rfi[i].size, rde.rfi[i].name);
*/
     }
     free(rde.rfi);
}


void usage_message(char * cmd_name)
{
/* clach04 - follow convertions for version output */
fprintf(stderr, "\n%s version %s Usage:\n", cmd_name, ftr_version);
fprintf(stderr, "\n  %s ", cmd_name);
/* list options */
fprintf(stderr, "[-v] ");
fprintf(stderr, "[-V] ");
fprintf(stderr, "[-l device] ");
fprintf(stderr, "[-b baudrate] ");
fprintf(stderr, "[non interactive commands]\n\n");

/* list explanations*/
fprintf(stderr, "Where:\n");
fprintf(stderr, "  -v/-V - small version info / big info\n");
fprintf(stderr, "  -l - specifies the device, e.g. -l /dev/ttyS0\
 (first com port)\n");
fprintf(stderr, "  -b  - specifies the baud rate\n");
fprintf(stderr, "   valid rates are: 19200, 38400 (default), 57600\n");
fprintf(stderr, "  commands - list of commands to then then quit\n");
fprintf(stderr, "              e.g. %s put test.exf\n\n", cmd_name);
fprintf(stderr, "Default is to open COM1 (/dev/ttyS0) with OnHand baud rate (38400)\n\n");
}

int
main(int argc, char *argv[])
{
     char	*name;
     int	i, c = 1, opts = 1;
     int        rn;  /* ruputer negotiate */
     char       connect_speed[20] = {"38400"};
     int        show_usage = 0;
     char       *tmp_envvar;

     tmp_envvar = getenv("FTON_SERIALPORT");
     if (tmp_envvar != NULL ) strcpy (rup_line,tmp_envvar);
     tmp_envvar = getenv("FTON_BAUDRATE");
     if (tmp_envvar != NULL ) strcpy (connect_speed,tmp_envvar);



     for (i = 1; i < argc; i++) {
	  name = argv[i];
	  if (*name == '-') {
	       opts++;
	       for (name++; *name != '\0'; name++) {
		    argv[i] = NULL;
		    if (c == i )
			 c++;
		    switch(*name) {
		    case 'V':
			 fprintf(stderr, "%s version %s\n", argv[0], ftr_version);
			 fprintf(stderr, "Copyright (C) 1998 N.Fukase\n");
			 fprintf(stderr, "All Rigth Reserverd\n");
			 return 0;
		    case 'v':
			 fprintf(stderr, "%s version %s\n", argv[0], ftr_version);
			 return 0;
		    case 'l':
			 if (name[1] == '\0') {
			      opts++;
			      i++;
			      strcpy(rup_line, argv[i]);
			      argv[i] = NULL;
			      if (c == i)
				   c++;
			      goto exit_loop;
			 } else {
			      strcpy(rup_line, name);
			      goto exit_loop;
			 }
			 break;
                    case 'b':
                         if (name[1] == '\0') {
                              opts++;
                              i++;
                              strcpy(connect_speed, argv[i]);
                              argv[i] = NULL;
                              if (c == i)
                                   c++;
                              goto exit_loop;
                         } else {
                              strcpy(rup_line, name);
                              goto exit_loop;
                         }
                         break;
		    default:
			 fprintf(stderr, "Invalid option letter %c\n", *name);
                         show_usage = 1;
			 break;
		    }
		      
	       }
	  exit_loop:
	  }
     }

     if (show_usage != 0 ) usage_message(argv[0]);

     if (argc == opts) {
	  fprintf(stderr, "Connecting onHand on %s....\n", rup_line);
          fprintf(stderr, "Trying to initialize line at %s baud...\n", connect_speed);

	  /* Open Serial port */
	  init_line(rup_line, connect_speed);

          fprintf(stderr, "Line initialized. (CTRL-C to abort)\n");

	  /* Connect */
	  while((rn = ruputer_negotiate()) < 0)
	    fprintf(stderr, "Failure: ruputer_negotiate = %d\n", rn);
	  fprintf(stderr, "Connect!\n");

	  /* Ensure we are in the root directory of the flash drive */
	  command_chdir("b:\\");

	   /* Get/Run commands loop until quit */
	  intteruct();

	  /* Disconnect */
	  command_end();
	  /* Close Serial */
	  restore_line();

	  return 0;
     } else {
	  for (i = 0; (name = commands[i].name) != NULL; i++) {
	       if ((strcmp(name, argv[c]) == 0) && (commands[i].func != NULL)){
		    init_line(rup_line, connect_speed);
		    while(ruputer_negotiate() < 0)
			 ;
		    command_chdir("b:\\");
		    (*(commands[i].func))(&argv[c]);
		    command_end();
		    restore_line();
		    return 0;
	       }
	  }
	  return -1;
     }
}

/* Remote ls (i.e. ls of Ruputer directory */
int	rlc_ls(char **coms)
{
     if(coms[1] == NULL)
	  get_file_list("*.*", rup_cwd);
     else
	  get_file_list(coms[1], rup_cwd);
     return 0;
}

/* Remote ls -l (i.e. ls -l of Ruputer directory */
int     rlc_lsl(char **coms)
{
     if(coms[1] == NULL)
          get_file_listlong("*.*", rup_cwd);
     else
          get_file_listlong(coms[1], rup_cwd);
     return 0;
}




/*_____ ver RuputerOSСɽ _____________________________________*/
int	rlc_ver(char **coms)
{
     int	r;

     r = command_ver();
     printf("%d.%d\n", (r & 0xff), ((r & 0xff00) >> 8));
     return 0;
}
/*_____ pwd RuputerCWDɽ ______________________________________________*/
int	rlc_pwd(char **coms)
{
     printf("onHand\'s Current Directory is %s\n", rup_cwd);
     return 0;
}
/*_____ ren RuputerΥե͡ ___________________________________*/
int	rlc_ren(char **coms)
{
     if ((coms[1] != NULL ) & (coms[2] != NULL)) {
	  command_ren(coms[2], coms[1]);
	  return 0;
     } else {
	  printf("Too few arguement\n");
	  return -1;
     }
}
/*_____ pwd RuputerΥե򥳥ԡ _____________________________________*/
int	rlc_cp(char **coms)
{
     if ((coms[1] != NULL ) & (coms[2] != NULL)) {
	  command_copy(coms[2], coms[1]);
	  return 0;
     } else {
	  printf("Too few arguement\n");
	  return -1;
     }
}
/*_____ mkdir Ruputer˥ǥ쥯ȥ ___________________________________*/
int	rlc_mkdir(char **coms)
{
     int	i;

     if (coms[1] != NULL) {
	  for (i = 1; coms[i] != NULL ; ++i) {
	       command_mkdir(coms[i]);
	  }
	  return 0;
     } else {
	  printf("Too few arguement\n");
	  return -1;
     }
}
/*_____ rmdir RuputerΥǥ쥯ȥ ___________________________________*/
int	rlc_rmdir(char **coms)
{
     int	i;

     if (coms[1] != NULL) {
	  for (i = 1; coms[i] != NULL ; ++i) {
	       command_rmdir(coms[i]);
	  }
	  return 0;
     } else {
	  printf("Too few arguement\n");
	  return -1;
     }
}
/*_____ del RuputerΥե _________________________________________*/
int	rlc_del(char **coms)
{
     int	i;

     if (coms[1] != NULL) {
	  for (i = 1; coms[i] != NULL ; ++i) {
	       command_del(coms[i]);
	  }
	  return 0;
     } else {
	  printf("Too few arguement\n");
	  return -1;
     }
}
/*_____ lpwd CWDɽ(lsؤwrapper) _________________________________*/
int	rlc_lpwd(char **coms)
{
     char buf[FILENAME_MAX];

     getcwd(buf, FILENAME_MAX);
     printf("Local current directory is %s\n", buf);
     return 0;
}
/*_____ lcd CWDư _________________________________________________*/
int	rlc_lcd(char **coms)
{
     char buf[FILENAME_MAX];

     if (coms[1] != NULL)
	  chdir(coms[1]);
     else
	  chdir(getenv("HOME"));
     getcwd(buf, FILENAME_MAX);
     printf("Now, local current directory is %s\n", buf);
     return 0;
}
/*_____ put Ruputerإեž ___________________________________________*/
int	rlc_put(char **coms)
{
     if (coms[1] != NULL)
	  send_file(coms[1], rup_cwd);
     return 0;
}
/*_____ get Ruputerեž _______________________________________*/
int	rlc_get(char **coms)
{
     if (coms[1] != NULL)
	  recv_file(coms[1], rup_cwd);
     return 0;
}

/* Local ls */
int	rlc_lls(char **coms)
{
     int	pid, status;

     if ((pid = fork()) < 0) {
	  perror("fork");
	  return -1;
     }
     if (pid == 0) {
	  strcpy(coms[0], "ls");
	  execvp(coms[0],  coms);
	  perror(coms[0]);
	  exit(-1);
     }
     while(wait(&status) != pid)
	  ;
     return status;
}
/*_____ cd RuputerCWDѹ _______________________________________________*/
int	rlc_cd(char **coms)
{
     char	*frm, *to, *rc, *sl, *nsl, *drv;
     RDENT	rde;

     rc  = NULL;  sl  = NULL;  // to stop the warnings.
     frm = coms[1];
     to = rde.path;

     if (frm == NULL)
	  return rlc_pwd(coms);
     if (frm[1] == ':') {
	  if ((frm[0] == 'A') || (frm[0] == 'a')) {
	       rc = drv = rup_cwdA;
	  } else if((frm[0] == 'B') || (frm[0] == 'b')) {
	       rc = drv = rup_cwdB;
	  } else {
	       printf("Invalid drive letter\n");
	       return -1;
	  }
	  frm += 2;
     } else {
	  if (*rup_cwd == 'A') {
	       rc = drv = rup_cwdA;
	  } else if(*rup_cwd == 'B') {
	       rc = drv = rup_cwdB;
	  }
	  strncpy(to, rc, 2);
     }
     strncpy(to, rc, 2);
     to += 2;
     rc += 2;
     nsl = NULL;
     if ((*frm != '\\') && (*frm != '/')) {
	  /* Хѥ */
	  while((*to = *rc) != '\0') {
	       if ((*to == '\\') || (*to == '/')) {
		    *to = '\\';
		    if(nsl  == NULL)
			 nsl = sl = to;
		    else {
			 sl = nsl;
			 nsl = to;
		    }
	       }
	       to++;
	       rc++;
	  }
	  if (to[-1] != '\\') {
	       *to = '\\';
	       if(nsl  == NULL)
		    nsl = sl = to;
	       else {
		    sl = nsl;
		    nsl = to;
	       }
	       to++;
	  }
     } else {
	  *to = '\\';
	  nsl = sl = to;
	  frm++;
	  to++;
     }
     while(*frm != '\0') {
	  if ((*frm == '.') && (frm[1] == '.')){
	       frm++;
	       if (to[-1] != '\\') 
		    to = nsl;
	       else {
		    to = sl;
		    *to = '\0';
		    nsl = sl;
		    sl = strrchr(rde.path, '\\');
		    *to = '\\';
		    to++;
		    if (sl == NULL)
			 sl = nsl;
	       }
	  } else {
	       if ((*frm == '/') || (*frm == '\\')) {
		    if (to[-1] == '\\') {
			 frm++;
			 continue;
		    }
		    *to = '\\';
		    if(nsl  == NULL)
			 nsl = sl = to;
		    else {
			 sl = nsl;
			 nsl = to;
		    }
		    sl = to;
	       }else
		    *to = *frm;
	       to++;
	  }
	  frm++;
     }
     *to = '\0';
     if((strcmp(rde.path, "A:\\") == 0) || (strcmp(rde.path, "B:\\") == 0)) {
	  printf("Now, onHand\'s current directory is %s\n", rde.path);
	  strcpy(rup_cwd, rde.path);
	  strcat(rup_cwd, "\\");
	  if (*rup_cwd == 'A')
	       strcpy(rup_cwdA, rde.path);
	  else if (*rup_cwd == 'B')
	       strcpy(rup_cwdB, rde.path);
     } else {
	  command_dir(&rde);
	  if(rde.nfiles == 1) {
	       printf("Now, onHand\'s current directory is %s\n", rde.path);
	       strcpy(rup_cwd, rde.path);
	       strcat(rup_cwd, "\\");
	       if (*rup_cwd == 'A')
		    strcpy(rup_cwdA, rde.path);
	       else if (*rup_cwd == 'B')
		    strcpy(rup_cwdB, rde.path);
	  }
     }
     return 0;
}
/*_____ 饤إפɽ ______________________________________________*/
int
rlc_help(char **coms)
{
     int	i;

     if(coms[1] != NULL) {
	  for(i = 0; commands[i].name != NULL; i++)  {
	       if (strcmp(commands[i].name, coms[1]) == 0) {
		    printf("%s\t%s\n", commands[i].name, commands[i].doc);
		    return 0;
	       }
	  }
     }
     for (i = 0; commands[i].name != NULL; ++i) {
	  printf("%s\t%s\n", commands[i].name, commands[i].doc);
     }
     return 0;
}
