/*
 * rupserial.c - low level serial routines
 * Copyright (C) 1998 N.Fukase
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA
 */

/*
** 10-March-2002 clach04
**  Added comms_speed to init_line() and parameter to control baud rate
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>

#include "rupcommand.h"
#include "rupserial.h"

/*_____ ץ饤١ѿ ____________________________________________________*/
static struct termios	term,
			saveterm;
static int	dev;
/*----- Ƽ拾ѥޥ --------------------------------------------------*/
/* Ruputer Rshell ꥶȥ */
#define RRESULT_ACK	0x06
#define RRESULT_NAK	0x15

/* CRLF */
#define LF	0x0a
#define CR	0x0d

/* XMODEM */
#define XMODEM_ACK      0x06
#define XMODEM_NAK      0x15
#define XMODEM_CAN      0x18
#define XMODEM_SOH      0x01
#define XMODEM_STX      0x02
#define XMODEM_EOT      0x04
#define XMODEM_EOF      0x1A
#define XMODEM_C        0x43
#define XMODEM_SP       0x20
#define XMODEM_SYN      0x16
#define XMODEM_DLE      0x10
#define XMODEM_XOF      0x11
#define XMODEM_XON      0x13
#define XMODEM_W        0x57
/*===== ֥롼 ========================================================*/
/*----- Ruputerackå ------------------------------------------*/
int
chk_ack(void)
{
  char b;

  read(dev, &b, 1);
#ifdef DEBUG
  printf("%02x\n" , b & 0xff);
#endif
  return (b != RRESULT_ACK) ? -1 : 0;
}
/*----- Ruputerack --------------------------------------------------*/
void
put_ack(void)
{
  char b;
  b = RRESULT_ACK;
  write(dev, &b, 1);
}
/*----- ꥢǥХν --------------------------------------------*/
int
init_line(char *line, char * line_speed)
{
     long	f;
     speed_t comms_speed;

    if (strcmp (line_speed, "38400") == 0)
     comms_speed = B38400;
    else if (strcmp (line_speed, "19200") == 0)
     comms_speed = B19200;
    else if (strcmp (line_speed, "57600") == 0)
     comms_speed = B57600;
    else
     comms_speed = B38400;

     if((dev = open(line, (O_RDWR | O_NOCTTY | O_NONBLOCK))) < 0) {
	  fprintf(stderr, "Cannot open %s\n", line);
	  exit(-1);
     }
     if (tcgetattr(dev, &term) == -1) {
	  fprintf(stderr, "Cannot get attribute %s\n", line);
	  close(dev);
	  exit(-1);
     }
     memcpy(&saveterm, &term, sizeof(struct termios));
	 
#ifndef MACOSX
     term.c_iflag &= ~(IGNBRK | IGNCR | INLCR | ICRNL | IXANY |
		       IXON | IXOFF | INPCK | ISTRIP);
     term.c_iflag |= BRKINT | IGNPAR;
     term.c_oflag &= ~OPOST;
     term.c_lflag = ~(ICANON | ISIG | ECHO | ECHONL | ECHOE | ECHOK);
	 term.c_cflag |= CS8 | CREAD;
#else
	 /* 8 Bit Words, Receivers on, Don't mind Flow Control, Ignore status lines */
	 term.c_cflag |= CS8 | CREAD | CCTS_OFLOW | CRTS_IFLOW | CLOCAL;
#endif

     /*  baud = B19200;
      *  baud = B38400;
      *  baud = B57600; 
      */
     cfsetospeed(&term, comms_speed);
     cfsetispeed(&term, comms_speed);
     tcsetattr(dev, TCSANOW, &term);
     f = fcntl(dev, F_GETFL);
     tcflush(dev, TCIOFLUSH);
     f &= ~O_NONBLOCK;
     fcntl(dev, F_SETFL, f);
     return 0;
}
/*_____ ꥢǥХθ _____________________________________*/
void restore_line(void)
{
     tcflush(dev, TCIOFLUSH);
     tcsetattr(dev, TCSANOW, &saveterm);
     close(dev);
}
/*_____ RuputerȤ³ͥ _______________________________*/
int
ruputer_negotiate(void)
{

/* 
 *   The Ruputer/OnHandPC transmits
 *	 the sequence '\r\n' every half
 *	 a second to indicate that it
 *	 is ready to received data. It waits
 *	 for the key sequence 'Erg' to be sent
 *	 to it. Upon receiving this, it transmits
 *	 '0x06' and then displays the 'Connected!'
 *	 message on its screen.
 */
	 char   b = 0;
     while(b != CR) {
	  read(dev, &b, 1);
//  This isn't necessary
//          printf("%02x\n" , b & 0xff);
     }
     while(b != LF) {
	  read(dev, &b, 1);
//  Neither is this
//		  printf("%02x\n" , b );
     }
     write(dev, "Erg", 3);
     read(dev, &b, 1);
	 
	 // Check to see if the buffer equals '0x06', if it
	 // does, then it should be initialized. Otherwise,
	 // better cycle back around and try again (unless
	 // tries are exhausted).
	 
     if(b == XMODEM_ACK)
	  return 0;
     else
	  return -1;
}
/*_____ X-MODEM(128SUM)ˤ륢åץ __________________________________*/
int
xmodem_up(LENDAT *data)
{
     unsigned char	buf[132], *ptr, bn, sum, ret;
     int		len, i;

     while((read(dev, &ret, 1) != 1) && (ret != XMODEM_NAK)) {
	  printf("%02x\n", ret);
     }
     fprintf(stderr, ".");
     len = data->length;
     ptr = data->data;
     buf[0] = XMODEM_SOH;
     for (bn = 1; len > 0; bn++) {
	  buf[1] = bn;    buf[2] = ~bn;
	  memcpy(buf + 3, ptr, (len > 128)? 128 : len);
	  len -= 128;
	  ptr += 128;
	  /* Ĥǡ128꾮 */
	  if (len < 0)
	       memset(buf + 3 + (128 + len), XMODEM_EOF, -len);
	  sum = 0;
	  for (i = 3; i < 128 + 3; i++)
	       sum += buf[i];
	  buf[131] = sum; 
	  for (ret = XMODEM_NAK; ret == XMODEM_NAK;) {
	       write(dev, buf, 132);
	       while(read(dev, &ret, 1) != 1)
		    ;
	       switch (ret){
	       case XMODEM_ACK:
		    fprintf(stderr, ".");
		    break;
	       case XMODEM_NAK:
		    fprintf(stderr, "x");
		    break;
	       case XMODEM_CAN:
		    fprintf(stderr, "\nTransfer cannceled by onHand\n");
		    return -1;
	       default:
		    fprintf(stderr, "\nInvalid result code from onHand\n");
		    return -1;
	       }
	  }
	  if(ret != XMODEM_ACK) {
	       printf("\nXMODEM: Bad Result Code\n");
	       return -1;
	  }
     }
     buf[0] = XMODEM_EOT;
     do {
	  write(dev, buf, 1);
	  while(read(dev, &ret, 1) != 1)
	       ;
	  printf("\n");
     } while (ret == XMODEM_NAK);
     if(ret != XMODEM_ACK) {
	  fprintf(stderr, "XMODEM: Bad Result Code %02x\n", ret & 0xff);
	  return -1;
     }
     return 0;
}
/*_____ X-MODEM(128SUM)ˤ ___________________________________*/
int
xmodem_down(LENDAT *data)
{
     unsigned char	buf[132], *ptr, bn, sum, ret;
     int		len, i, c;

     /*  */
     buf[0] = XMODEM_NAK;
     for (;;) {
	  write(dev, buf, 1);
	  while(read(dev, &ret, 1) != 1)
	       ;
	  if (ret == XMODEM_SOH)
	       break;
	  sleep(10);
     }
     /* ž */
     fprintf(stderr, ".");
     len = data->length;
     ptr = data->data;
     for (bn = 1; len > 0; ) {
	  if(ret != XMODEM_SOH) {
	       /* ǡ? */
	       buf[0] = XMODEM_CAN;
	       write(dev, buf, 1);
	       fprintf(stderr, "\nTransfer failed\n");
	       return -1;
	  }
	  for (c = 0; c < 131;) {
	       c += read(dev, buf + c, 131 - c);
	  }
	  if((buf[0] != bn) || (buf[1] != (~bn & 0xff))) {
	       buf[0] = XMODEM_NAK;
	       fprintf(stderr, "X");
	       write(dev, buf, 1);
	       while(read(dev, &ret, 1) != 1)
		    ;
	       continue;
	  }
	  /* SUM å */
	  sum = 0;
	  for (i = 2; i < 128 + 2; i++)
	       sum += buf[i];
	  if (buf[130] != sum) {
	       buf[0] = XMODEM_NAK;
	       fprintf(stderr, "x");
	       write(dev, buf, 1);
	       while(read(dev, &ret, 1) != 1)
		    ;
	       continue;
	  }
	  memcpy(ptr, buf + 2, (len > 128)? 128 : len);
	  buf[0] = XMODEM_ACK;
	  write(dev, buf, 1);
	  fprintf(stderr, ".");
	  len -= 128;
	  ptr += 128;
	  while(read(dev, &ret, 1) != 1)
	       ;
	  bn++;
     }
     printf("\n");
     if (ret == XMODEM_EOT) {
	  buf[0] = XMODEM_ACK;
	  write(dev, buf, 1);
	  return 0;
     } else {
	  buf[0] = XMODEM_CAN;
	  write(dev, buf, 1);
	  fprintf(stderr, "At last, transfer failed\n");
	  return -1;
     }
}
/*_____ RuputerؤΥޥȯ _____________________________________________*/
int
ruputer_command(int command, int sub, int length, char *data)
{
     int	i, sum;
     char	*buf;
  
     if((buf = (char *)malloc(length + 1)) == NULL) {
	  fprintf(stderr, "Cannot allocate memory!\n");
	  return -1;
     }
     buf[0] = (command & 0xff);
     buf[1] = (sub & 0xff);
     WORD2(buf + 2, length);
     if (length > 4) {
	  memcpy(buf + 4, data, (length - 4));
     }
     sum = 0;
     for(i = 0; i < length; ++i) {
	  sum += (buf[i] & 0xff);
     }
     buf[length] = ((~sum) & 0xff);
     write(dev, buf, length + 1);
     free(buf);
     return 0;
}
/*_____ RuputerΥ쥹ݥ󥹤å ___________________________________*/
int
ruputer_response(int res, int sub, LENDAT *dat)
{
     char	b[4];
     int	c, length, sum, i;

     dat->data = NULL;
     /* get header */
     for (c = 0; c < 4;) {
	  c += read(dev, b + c, 4 - c);
     }
     /* check header */
     if ((b[0] != res) || (b[1] != sub))
	  return -1;
     length = WORD(b + 2);
     /* make return structure */
     if (length > 4) {
	  length -= 4;
	  dat->length = length;
	  if((dat->data = (char *)malloc(length)) == NULL) {
	       fprintf(stderr, "Cannot allocate memory\n" );
	       return -1;
	  }
	  /* get data */
	  for (c = 0; c < length;) {
	       c += read(dev, dat->data + c, length - c);
	  }
     }
     /* sum check */
     sum = (b[0] & 0xff) + (b[1] & 0xff) + (b[2] & 0xff) + (b[3] & 0xff);
     for(i = 0; i < length; ++i) {
	  sum += (dat->data[i] & 0xff);
     }
     while(read(dev, b, 1) < 1)
	  ;
     if (((~sum) & 0xff) == (*b & 0xff))
	  return 0;
     if (dat) {
	  free(dat->data);
     }
     return -1;
}
